

--
CREATE OR REPLACE FUNCTION TArtikel.opl__op_ix__get__standard_by__aknr(IN _aknr varchar)
    RETURNS INTEGER
    AS $$
        SELECT op_ix FROM opl WHERE op_n = _aknr AND op_standard;
    $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;


-- Funktion zum Kopieren von alternativen Kostenstellen und NC-Programmen
CREATE OR REPLACE FUNCTION tartikel.opl__copy__ask__copy__op2ks__ncp(
      _src_op_ix    integer, -- ASK-Index der Quelle
      _trg_op_ix    integer  -- ASK-Index des Ziels
) RETURNS VOID AS $$
  DECLARE
    _r RECORD;
    _src_o2_id integer;
    _trg_o2_id integer;
  BEGIN

    -- Keine Arbeitsgänge? Dann gibt es nichts zu tun.
    IF NOT EXISTS( SELECT 1 FROM op2 WHERE o2_ix = _src_op_ix ) THEN
      RETURN;
    END IF;

    FOR _r IN ( SELECT * FROM op2 WHERE o2_ix = _src_op_ix ) LOOP

      _src_o2_id := _r.o2_id;
      _trg_o2_id := o2_id FROM op2 WHERE o2_ix = _trg_op_ix AND o2_n = _r.o2_n;

      -- Kopie alternative Kostenstelle
      INSERT INTO op2ksa
           ( o2ks_o2_id, o2ks_ks, o2ks_ksap, o2ks_ba, o2ks_txt, o2ks_ks_id, o2ks_ks_ba_id )
      SELECT _trg_o2_id, o2ks_ks, o2ks_ksap, o2ks_ba, o2ks_txt, o2ks_ks_id, o2ks_ks_ba_id
      FROM op2ksa WHERE o2ks_o2_id = _src_o2_id;

      -- Kopie NC-Programme
      INSERT INTO ncp_o2_a2_ksv
            ( ncpoak_ncp_id, ncpoak_o2_id, ncpoak_a2_id, ncpoak_ks, ncpoak_txt )
      SELECT  ncpoak_ncp_id,   _trg_o2_id, ncpoak_a2_id, ncpoak_ks, ncpoak_txt
      FROM ncp_o2_a2_ksv WHERE ncpoak_o2_id = _src_o2_id;

    END LOOP;

  END $$ LANGUAGE plpgsql VOLATILE;
--
--
--- #17190 ASK kopieren durch einzelne Funktionen
--- Logik:
---        Von Quelledatensatz wird JSON für neuer Datensatz aufgebaut
---        Im JSON Value für notwendige Felder werden angepasst
---        von JSON INSERT-Statament erstellen und einspielen
---        Kontrolle, ob Quelle- und Zieldatensatzanzahl stimmt

CREATE OR REPLACE FUNCTION TArtikel.copy_ASK__op2__duplicate(
      IN _src_op_ix    integer,        -- ASK-Index der Quelle
      IN _trg_op_ix    integer         -- ASK-Index des Ziels
  ) RETURNS void AS $$
  DECLARE  _serial_pk   integer := 0;  --- Serial Key - Zieldatensatz
           _json        json;
           _tmp         record;
           _sql         text;
           rec          record;
  BEGIN
    FOR rec IN ( SELECT * FROM op2 WHERE o2_ix = _src_op_ix ) LOOP

        _serial_pk = nextval('op2_o2_id_seq');

        --- JSON erstellen
        SELECT * FROM op2 WHERE o2_id = rec.o2_id INTO _tmp;
        SELECT row_to_json( _tmp ) INTO _json;

        --- neue Werte in JSON übernehmen
        SELECT tlog.json__value__replace( _json, 'o2_id' , _serial_pk::varchar, 'integer' ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o2_ix' , _trg_op_ix::varchar, 'integer' ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o2_nc' , 'null'             , 'null'    ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o2_nc2', 'null'             , 'null'    ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o2_nc3', 'null'             , 'null'    ) INTO _json;

        --- Modified-Spalten
        SELECT tlog.json__value__dbrid_insert_modified__replace( _json ) INTO _json;

        --- INSERT-Statament erstellen
        SELECT tlog.Auditlog__json_old__into_sql( 'op2', _json ) INTO _sql;
        --- INSERT-Statament ausführen
        EXECUTE _sql ;
        ---
    END LOOP;

    --- Kontrolle
    PERFORM tsystem.selects_count__control( 'op2',
                                            'SELECT count(*) FROM op2 WHERE o2_ix = ' || _src_op_ix || ';',
                                            'SELECT count(*) FROM op2 WHERE o2_ix = ' || _trg_op_ix || ';' );
  END $$ LANGUAGE plpgsql VOLATILE;
---
CREATE OR REPLACE FUNCTION TArtikel.copy_ASK__op5__duplicate(
      IN _src_op_ix    integer,        -- ASK-Index der Quelle
      IN _trg_op_ix    integer         -- ASK-Index des Ziels
  ) RETURNS void AS $$
  DECLARE  _serial_pk      integer := 0;  --- Serial Key - Zieldatensatz
           _new_o5_o2_id   integer := 0;
           _new_o2_id      integer := 0;
           _json           json;
           _tmp            record;
           _sql            text;
           rec             record;
  BEGIN

    FOR rec IN ( SELECT o5_id, o5_o2_id, o2_ix, o2_n
                 FROM op5
                   JOIN op2 ON o2_id = o5_o2_id
                 WHERE o2_ix = _src_op_ix
               ) LOOP

        _serial_pk = nextval('op5_o5_id_seq');
        SELECT o2_id FROM op2 WHERE o2_ix = _trg_op_ix AND o2_n = rec.o2_n INTO _new_o2_id;

        --- JSON erstellen
        SELECT * FROM op5 WHERE o5_id = rec.o5_id INTO _tmp;
        SELECT row_to_json( _tmp ) INTO _json;

        --- neue Werte in JSON übernehmen
        SELECT tlog.json__value__replace( _json, 'o5_id'   , _serial_pk::varchar, 'integer' ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o5_o2_id', _new_o2_id::varchar, 'integer' ) INTO _json;

        --- Modified-Spalten
        SELECT tlog.json__value__dbrid_insert_modified__replace( _json ) INTO _json;

        --- INSERT-Statament erstellen
        SELECT tlog.Auditlog__json_old__into_sql( 'op5', _json ) INTO _sql;
        --- INSERT-Statament ausführen
        EXECUTE _sql ;

    END LOOP;

    --- Kontrolle
    PERFORM tsystem.selects_count__control(
                                            'op5',
                                            'SELECT count(*) FROM ( SELECT o2_ix, o2_n, op5.* FROM op2 JOIN op5 ON o2_id = o5_o2_id WHERE o2_ix = ' || _src_op_ix || ' ) AS sub;'
                                            ,
                                            'SELECT count(*) FROM ( SELECT o2_ix, o2_n, op5.* FROM op2 JOIN op5 ON o2_id = o5_o2_id WHERE o2_ix = ' || _trg_op_ix || ' ) AS sub;'
                                          );
  END $$ LANGUAGE plpgsql VOLATILE;
---
CREATE OR REPLACE FUNCTION TArtikel.copy_ASK__oplpm__duplicate(
      IN _src_op_ix    integer,        -- ASK-Index der Quelle
      IN _trg_op_ix    integer         -- ASK-Index des Ziels
  ) RETURNS void AS $$
  DECLARE  _serial_pk   integer := 0;  --- Serial Key - Zieldatensatz
           _json        json;
           _tmp         record;
           _sql         text;
           rec          record;
           _o2_id_new   integer;
  BEGIN

    FOR rec IN ( SELECT  o2_ix, o2_n, o2_id, oplpm.*
                 FROM oplpm
                   JOIN op2 ON o2_id = pm_op2_id AND o2_ix = _src_op_ix
               ) LOOP

        _serial_pk = nextval( 'oplpm_data_pm_id_seq' );

        SELECT o2_id FROM op2 WHERE o2_ix = _trg_op_ix AND o2_n = rec.o2_n INTO _o2_id_new;

        --- JSON erstellen
        SELECT * FROM oplpm WHERE pm_id = rec.pm_id INTO _tmp;
        SELECT row_to_json( _tmp ) INTO _json;

        --- neue Werte in JSON übernehmen
        SELECT tlog.json__value__replace( _json, 'pm_id'     , _serial_pk::varchar, 'integer' ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'pm_op2_id' , _o2_id_new::varchar, 'integer' ) INTO _json;

        --- Modified-Spalten
        SELECT tlog.json__value__dbrid_insert_modified__replace( _json ) INTO _json;

        --- INSERT-Statament erstellen
        SELECT tlog.Auditlog__json_old__into_sql( 'oplpm', _json ) INTO _sql;
        --- INSERT-Statament ausführen
        EXECUTE _sql ;
        ---
    END LOOP;
    --- Kontrolle
    PERFORM tsystem.selects_count__control( 'oplpm',
                                            'SELECT count(*) FROM ( SELECT o2_ix, o2_n, oplpm.* FROM oplpm JOIN op2 ON o2_id = pm_op2_id WHERE o2_ix = ' || _src_op_ix || ' ) AS sub;'
                                            ,
                                            'SELECT count(*) FROM ( SELECT o2_ix, o2_n, oplpm.* FROM oplpm JOIN op2 ON o2_id = pm_op2_id WHERE o2_ix = ' || _trg_op_ix || ' ) AS sub;' );
    -- ASK QS-Freigabe
    UPDATE opl SET op_qs_freigabe = false
    WHERE op_ix = _trg_op_ix AND EXISTS(SELECT true FROM oplpm WHERE pm_op2_id = _trg_op_ix AND pm_op2_id IS NOT NULL);

  END $$ LANGUAGE plpgsql VOLATILE;
---
CREATE OR REPLACE FUNCTION TArtikel.copy_ASK__op3__duplicate(
      IN _src_op_ix    integer,        -- ASK-Index der Quelle
      IN _trg_op_ix    integer         -- ASK-Index des Ziels
  ) RETURNS void AS $$
  DECLARE  _serial_pk   integer := 0;  --- Serial Key - Zieldatensatz
           _json        json;
           _tmp         record;
           _sql         text;
           rec          record;
  BEGIN
    ---
    DELETE FROM op3 WHERE o3_ix = _trg_op_ix;

    FOR rec IN ( SELECT * FROM op3 WHERE o3_ix = _src_op_ix ) LOOP

        _serial_pk = nextval('op3_o3_id_seq');

        --- JSON erstellen
        SELECT * FROM op3 WHERE o3_id = rec.o3_id INTO _tmp;
        SELECT row_to_json( _tmp ) INTO _json;

        --- neue Werte in JSON übernehmen
        SELECT tlog.json__value__replace( _json, 'o3_id' , _serial_pk::varchar, 'integer' ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o3_ix' , _trg_op_ix::varchar, 'integer' ) INTO _json;

        --- Modified-Spalten
        SELECT tlog.json__value__dbrid_insert_modified__replace( _json ) INTO _json;

        --- INSERT-Statament erstellen
        SELECT tlog.Auditlog__json_old__into_sql( 'op3', _json ) INTO _sql;
        --- INSERT-Statament ausführen
        EXECUTE _sql ;

    END LOOP;

    --- Kontrolle
    PERFORM tsystem.selects_count__control( 'op3',
                                            'SELECT count(*) FROM op3 WHERE o3_ix = ' || _src_op_ix || ';',
                                            'SELECT count(*) FROM op3 WHERE o3_ix = ' || _trg_op_ix || ';' );
  END $$ LANGUAGE plpgsql VOLATILE;
---
CREATE OR REPLACE FUNCTION TArtikel.copy_ASK__op6__duplicate(
      IN _src_op_ix    integer,        -- ASK-Index der Quelle
      IN _trg_op_ix    integer         -- ASK-Index des Ziels
  ) RETURNS void AS $$
  DECLARE  _serial_pk   integer := 0;  --- Serial Key - Zieldatensatz
           _json        json;
           _tmp         record;
           _sql         text;
           rec          record;
  BEGIN
    ---
    DELETE FROM op6 WHERE o6_ix = _trg_op_ix;

    FOR rec IN ( SELECT * FROM op6 WHERE o6_ix = _src_op_ix ) LOOP

        _serial_pk = nextval('op6_o6_id_seq');

        --- JSON erstellen
        SELECT * FROM op6 WHERE o6_id = rec.o6_id INTO _tmp;
        SELECT row_to_json( _tmp ) INTO _json;

        --- neue Werte in JSON übernehmen
        SELECT tlog.json__value__replace( _json, 'o6_id' , _serial_pk::varchar, 'integer' ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o6_ix' , _trg_op_ix::varchar, 'integer' ) INTO _json;

        --- Modified-Spalten
        SELECT tlog.json__value__dbrid_insert_modified__replace( _json ) INTO _json;

        --- INSERT-Statament erstellen
        SELECT tlog.Auditlog__json_old__into_sql( 'op6', _json ) INTO _sql;
        --- INSERT-Statament ausführen
        EXECUTE _sql ;
        ---
    END LOOP;

    --- Kontrolle
    PERFORM tsystem.selects_count__control( 'op6',
                                            'SELECT count(*) FROM op6 WHERE o6_ix = ' || _src_op_ix || ';',
                                            'SELECT count(*) FROM op6 WHERE o6_ix = ' || _trg_op_ix || ';' );
  END $$ LANGUAGE plpgsql VOLATILE;
---
CREATE OR REPLACE FUNCTION TArtikel.copy_ASK__op2ba__duplicate(
      IN _src_op_ix    integer,        -- ASK-Index der Quelle
      IN _trg_op_ix    integer         -- ASK-Index des Ziels
  ) RETURNS void AS $$
  DECLARE  _serial_pk   integer := 0;  --- Serial Key - Zieldatensatz
           _json        json;
           _tmp         record;
           _sql         text;
           rec          record;
  BEGIN
    ---
    DELETE FROM op2ba WHERE o2ba_ix = _trg_op_ix;

    FOR rec IN ( SELECT * FROM op2ba WHERE o2ba_ix = _src_op_ix ) LOOP

        _serial_pk = nextval('op2ba_o2ba_id_seq');

        --- JSON erstellen
        SELECT * FROM op2ba WHERE o2ba_id = rec.o2ba_id INTO _tmp;
        SELECT row_to_json( _tmp ) INTO _json;

        --- neue Werte in JSON übernehmen
        SELECT tlog.json__value__replace( _json, 'o2ba_id' , _serial_pk::varchar, 'integer' ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o2ba_ix' , _trg_op_ix::varchar, 'integer' ) INTO _json;

        --- Modified-Spalten
        SELECT tlog.json__value__dbrid_insert_modified__replace( _json ) INTO _json;

        --- INSERT-Statament erstellen
        SELECT tlog.Auditlog__json_old__into_sql( 'op2ba', _json ) INTO _sql;
        --- INSERT-Statament ausführen
        EXECUTE _sql ;
        ---
    END LOOP;
    --- Kontrolle
    PERFORM tsystem.selects_count__control( 'op2ba',
                                            'SELECT count(*) FROM op2ba WHERE o2ba_ix = ' || _src_op_ix || ';',
                                            'SELECT count(*) FROM op2ba WHERE o2ba_ix = ' || _trg_op_ix || ';' );
  END $$ LANGUAGE plpgsql VOLATILE;
---
CREATE OR REPLACE FUNCTION TArtikel.copy_ASK__op7__duplicate(
      IN _src_op_ix    integer,        -- ASK-Index der Quelle
      IN _trg_op_ix    integer         -- ASK-Index des Ziels
  ) RETURNS void AS $$
  DECLARE  _serial_pk   integer := 0;  --- Serial Key - Zieldatensatz
           _json        json;
           _tmp         record;
           _sql         text;
           rec          record;
  BEGIN
    ---
    DELETE FROM op7 WHERE o7_op_ix = _trg_op_ix;

    FOR rec IN ( SELECT * FROM op7 WHERE o7_op_ix = _src_op_ix ) LOOP

        _serial_pk = nextval('op7_o7_id_seq');

        --- JSON erstellen
        SELECT * FROM op7 WHERE o7_id = rec.o7_id INTO _tmp;
        SELECT row_to_json( _tmp ) INTO _json;

        --- neue Werte in JSON übernehmen
        SELECT tlog.json__value__replace( _json, 'o7_id'   , _serial_pk::varchar, 'integer' ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o7_op_ix', _trg_op_ix::varchar, 'integer' ) INTO _json;

        --- Modified-Spalten
        SELECT tlog.json__value__dbrid_insert_modified__replace( _json ) INTO _json;

        --- INSERT-Statament erstellen
        SELECT tlog.Auditlog__json_old__into_sql( 'op7', _json ) INTO _sql;
        --- INSERT-Statament ausführen
        EXECUTE _sql ;
        ---
    END LOOP;

    --- Kontrolle
    PERFORM tsystem.selects_count__control( 'op7',
                                            'SELECT count(*) FROM op7 WHERE o7_op_ix = ' || _src_op_ix || ';',
                                            'SELECT count(*) FROM op7 WHERE o7_op_ix = ' || _trg_op_ix || ';' );
  END $$ LANGUAGE plpgsql VOLATILE;
---
CREATE OR REPLACE FUNCTION TArtikel.copy_ASK__op7zko__duplicate(
      IN _src_op_ix    integer,        -- ASK-Index der Quelle
      IN _trg_op_ix    integer         -- ASK-Index des Ziels
  ) RETURNS void AS $$
  DECLARE  _serial_pk   integer := 0;  --- Serial Key - Zieldatensatz
           _json        json;
           _tmp         record;
           _sql         text;
           rec          record;
  BEGIN

    DELETE FROM op7zko WHERE o7zk_ix = _trg_op_ix;

    FOR rec IN ( SELECT * FROM op7zko WHERE o7zk_ix = _src_op_ix ) LOOP

        _serial_pk = nextval('op7zko_o7zk_id_seq');

        --- JSON erstellen
        SELECT * FROM op7zko WHERE o7zk_id = rec.o7zk_id INTO _tmp;
        SELECT row_to_json( _tmp ) INTO _json;

        --- neue Werte in JSON übernehmen
        SELECT tlog.json__value__replace( _json, 'o7zk_id', _serial_pk::varchar, 'integer' ) INTO _json;
        SELECT tlog.json__value__replace( _json, 'o7zk_ix', _trg_op_ix::varchar, 'integer' ) INTO _json;

          -- System-IDs
        SELECT tlog.json__value__dbrid_insert_modified__replace( _json ) INTO _json;

        --- INSERT-Statament erstellen
        SELECT tlog.Auditlog__json_old__into_sql( 'op7zko', _json ) INTO _sql;
        --- INSERT-Statament ausführen
        EXECUTE _sql ;
        ---
    END LOOP;

    --- Kontrolle
    PERFORM tsystem.selects_count__control( 'op7zko',
                                            'SELECT count(*) FROM op7zko WHERE o7zk_ix = ' || _src_op_ix || ';',
                                            'SELECT count(*) FROM op7zko WHERE o7zk_ix = ' || _trg_op_ix || ';' );
  END $$ LANGUAGE plpgsql VOLATILE;
---
CREATE OR REPLACE FUNCTION TArtikel.copy_ASK__stv_op2__duplicate(
      IN _src_op_ix    integer,        -- ASK-Index der Quelle
      IN _trg_op_ix    integer,        -- ASK-Index des Ziels
      IN _src_ak_nr    varchar,        -- Artikel-Nr. der Quelle
      IN _trg_ak_nr    varchar         -- Artikel-Nr. des Ziels

  ) RETURNS void AS $$
  DECLARE
      -- Kopie Verknüpfung Arbeitsgang zu Stücklistenposition (stv_op2) durchführen.
      _do_stv_op2_copy  boolean;

      _serial_pk        integer := 0;
      _id               integer := 0;
      _json             json;
      _tmp              record;
      _sql              text;
      rec               record;
  BEGIN

        -- Verknüpfung Arbeitsgang zu Stücklistenposition (stv_op2)
        --- neuen Artikel löschen
        DELETE FROM stv_op2 WHERE sto2_st_id IN ( SELECT st_id FROM stv WHERE st_zn = _trg_ak_nr );

        -- Ermittlung ob Kopie durchgeführt werden kann.
        _do_stv_op2_copy := false;

        -- Fall 1: Quell- und Zielartikel sind gleich, d.h. es wird bloß andere ASK-Variante kopiert.
        IF _src_ak_nr = _trg_ak_nr THEN
            _do_stv_op2_copy := true;

        -- Fall 2: Es existieren für Quell- und Zielartikel identische Stücklisten
        -- Insb. Kopie von Verknüpfungen ausschließen bei:
          -- z.B. Artikel-Kopie mit ASK-Kopie ohne Stücklisten-Kopie.
          -- oder ASK-Kopie eines anderen Artikels in diesen mit anderer Stückliste
        ELSIF     EXISTS(SELECT true FROM stv WHERE st_zn = _src_ak_nr)
              AND EXISTS(SELECT true FROM stv WHERE st_zn = _trg_ak_nr)
        THEN
            -- Nur wenn Stücklisten identisch sind.
            _do_stv_op2_copy :=
                NOT EXISTS(
                    -- es gibt keine Position
                    SELECT true
                    -- aus beiden Stücklisten
                    FROM (
                        SELECT
                          st_n, st_pos
                        FROM stv AS src_stv
                        WHERE src_stv.st_zn = _src_ak_nr

                        UNION ALL
                        SELECT
                          st_n, st_pos
                        FROM stv AS trg_stv
                        WHERE trg_stv.st_zn = _trg_ak_nr
                    ) AS both_stv
                    -- die bloß in einer Stückliste vorhanden ist.
                    GROUP BY st_n, st_pos
                    HAVING count(*) < 2
                )
            ;

        END IF;


        -- Kopie von Arbeitsgang zu Stücklistenposition kann durchgeführt werden.
        IF _do_stv_op2_copy THEN

            -- Verknüpfungen von STV zu AG kopieren
            FOR rec IN (
                         SELECT o2_id, o2_ix, o2_n, stv_op2.*
                         FROM stv_op2
                            JOIN op2 ON o2_id = sto2_o2_id
                         WHERE o2_ix = _src_op_ix
                           AND sto2_st_id IS NOT null
                         ORDER BY sto2_id
                       ) LOOP

                _serial_pk = nextval('stv_op2_sto2_id_seq');

                --- JSON erstellen
                SELECT * FROM stv_op2 WHERE sto2_id = rec.sto2_id INTO _tmp;
                SELECT row_to_json( _tmp ) INTO _json;

                --- neue Werte in JSON übernehmen
                SELECT tlog.json__value__replace( _json, 'sto2_id'   , _serial_pk::varchar, 'integer' ) INTO _json;

                SELECT st_id FROM stv WHERE st_zn = _trg_ak_nr INTO _id;                                               --- ID stv Ziel
                SELECT tlog.json__value__replace( _json, 'sto2_st_id', _id::varchar, 'integer' ) INTO _json;
                ---

                SELECT o2_id FROM op2 WHERE o2_ix = _trg_op_ix AND o2_n = rec.o2_n INTO _id;                       --- ID ASK-AG Ziel
                SELECT tlog.json__value__replace( _json, 'sto2_o2_id', _trg_op_ix::varchar, 'integer' ) INTO _json;

                --- Modified-Spalten
                SELECT tlog.json__value__dbrid_insert_modified__replace( _json ) INTO _json;

                --- INSERT-Statament erstellen
                SELECT tlog.Auditlog__json_old__into_sql( 'stv_op2', _json ) INTO _sql;
                --- INSERT-Statament ausführen
                EXECUTE _sql ;
                ---
            END LOOP;

            PERFORM tsystem.selects_count__control( 
                'stv_op2',
                'SELECT count(*) FROM stv_op2 WHERE sto2_st_id IN ( SELECT st_id FROM stv WHERE st_zn = ' || quote_literal( _src_ak_nr ) || ' );',
                'SELECT count(*) FROM stv_op2 WHERE sto2_st_id IN ( SELECT st_id FROM stv WHERE st_zn = ' || quote_literal( _trg_ak_nr ) || ' );' 
            );
        END IF;
    --
  END $$ LANGUAGE plpgsql VOLATILE;
---
CREATE OR REPLACE FUNCTION TArtikel.opl__copy__ASK(
     IN  _src_op_ix    integer,        -- ASK-Index der Quelle
     IN  _trg_ak_nr    varchar,        -- Artikel-Nr. des Ziels
     IN  _trg_op_vi    varchar = null, -- ASK-Variante des Ziels
     IN  _trg_op_ix    integer = null, -- ASK-Index des Ziels
     IN  _chk_fert     boolean = true  -- Kopie soll Standard-Fertigungsvariante werden
  ) RETURNS integer AS $$
  DECLARE
      -- Artikel-Nr. der Quelle
      _src_ak_nr        varchar;
      -- Kopie Verknüpfung Arbeitsgang zu Stücklistenposition (stv_op2) durchführen.
      _do_stv_op2_copy  boolean;
      --- neue Variablen
      _json             json;
      _tmp              record;
      _sql              text;
  BEGIN
    -- Führt eine komplette Kopie einer AVOR-Stammkarte inkl. aller Unterelemente (siehe unten) durch.
      -- _src_op_ix: Quelle ist eine spez. ASK.
      -- _trg_ak_nr: Ziel ist spez. Artikel.
      -- _trg_op_vi (optional): neue Variante.    Wird autom. ermittelt, wenn nicht angg..
      -- _trg_op_ix (optional): neuer ASK-Index.  Wird autom. ermittelt, wenn nicht angg..
      -- _chk_fert (defaul true): Die Kopie soll die (neue) Standard-Fertigungsvariante werden.

    /* debug
      RAISE WARNING '_src_op_ix => %;
                     _trg_ak_nr => %;
                     _trg_op_vi => %;
                     _trg_op_ix => %;
                     _chk_fert  => %;
                     ',_src_op_ix
                      ,_trg_ak_nr
                      ,_trg_op_vi
                      ,_trg_op_ix
                      ,_chk_fert
      ;  
    */

    -- ohne Angabe von Quelle (ASK) oder Zielartikel raus
    IF ( _src_op_ix IS NULL OR _trg_ak_nr IS NULL ) THEN
        RETURN null;
    END IF;

    -- ohne Existenz von Quelle oder Zielartikel raus
    IF  (
              NOT EXISTS(SELECT true FROM opl WHERE op_ix = _src_op_ix)
          OR  NOT EXISTS(SELECT true FROM art WHERE ak_nr = _trg_ak_nr)
        )
    THEN
        RETURN null;
    END IF;


    _src_ak_nr := op_n FROM opl WHERE op_ix = _src_op_ix;

    -- autom. Ermittlung der neuen ASK-Variante
    -- Nur numerische Varianten können behandelt werden: +1.
    -- alphanumerische erhalten autom. die '1'
    IF _trg_op_vi IS NULL THEN
        _trg_op_vi := max( op_vi::numeric ) FROM opl WHERE op_n = _trg_ak_nr AND isNumeric( op_vi );
        _trg_op_vi := coalesce( _trg_op_vi, '0' )::numeric + 1;
    END IF;

    -- autom. Ermittlung des neuen ASK-Indexes
    IF _trg_op_ix IS NULL THEN
        _trg_op_ix := nextval('opl_op_ix_seq');
    END IF;

    -- bereits vorhandene Ziel-ASK ohne AG wird gelöscht
    DELETE FROM opl WHERE op_n = _trg_ak_nr AND op_vi = _trg_op_vi;

    -- ASK-Kopf (opl)
        --- JSON erstellen
        SELECT * FROM opl WHERE op_ix = _src_op_ix INTO _tmp;
        SELECT row_to_json( _tmp ) INTO _json;

        --- neue Werte in JSON übernehmen
        SELECT tlog.json__value__replace( _json, 'op_n', _trg_ak_nr, 'varchar' ) INTO _json;
        IF _trg_op_vi IS NOT null THEN
            SELECT tlog.json__value__replace( _json, 'op_vi', _trg_op_vi, 'varchar' ) INTO _json;
        END IF;
        SELECT tlog.json__value__replace( _json, 'op_ix', _trg_op_ix::varchar, 'integer' ) INTO _json;
        ---

        --- INSERT-Statament erstellen
        SELECT tlog.Auditlog__json_old__into_sql( 'opl', _json ) INTO _sql;
        --- INSERT-Statament ausführen
        EXECUTE _sql ;

    -- Arbeitsgänge (op2)
       PERFORM TArtikel.copy_ASK__op2__duplicate( _src_op_ix, _trg_op_ix );

    -- Verknüpfung Arbeitsgang zu Stücklistenposition (stv_op2)
        PERFORM TArtikel.copy_ASK__stv_op2__duplicate( _src_op_ix, _trg_op_ix, _src_ak_nr, _trg_ak_nr );

    -- Arbeitsschritte (op5)
        PERFORM TArtikel.copy_ASK__op5__duplicate( _src_op_ix, _trg_op_ix );

    -- Prüfprotokoll (oplpm)
        PERFORM TArtikel.copy_ASK__oplpm__duplicate( _src_op_ix, _trg_op_ix );

    -- ASK QS-Freigabe
        UPDATE opl SET op_qs_freigabe = false
        WHERE op_ix = _trg_op_ix AND EXISTS( SELECT true
                                             FROM oplpm
                                               JOIN op2 ON o2_id = pm_op2_id AND o2_ix = _trg_op_ix
                                             LIMIT 1
                                           );

    -- Sondereinzelkosten (op3)
        PERFORM TArtikel.copy_ASK__op3__duplicate( _src_op_ix, _trg_op_ix );

    -- Material (op6)
        PERFORM TArtikel.copy_ASK__op6__duplicate( _src_op_ix, _trg_op_ix );

    -- Auftrags- / Fertigungs-begleitende Artikel (op2ba)
        PERFORM TArtikel.copy_ASK__op2ba__duplicate( _src_op_ix, _trg_op_ix );

    -- Kalkulationsstaffeln (op7)
        PERFORM TArtikel.copy_ASK__op7__duplicate( _src_op_ix, _trg_op_ix );

    -- globale Zuschläge (op7zko)
        PERFORM TArtikel.copy_ASK__op7zko__duplicate( _src_op_ix, _trg_op_ix );

    -- Alternativkostenstellen mit NC-Programmen
        PERFORM tartikel.opl__copy__ask__copy__op2ks__ncp( _src_op_ix, _trg_op_ix );

    RETURN _trg_op_ix;
  END $$ LANGUAGE plpgsql VOLATILE;
--

--- #7372 Sollzeiten AVOR-Stammkarte [OG]
CREATE OR REPLACE FUNCTION TArtikel.op2__o2_tr__o2_th__update(
    IN in_a2_id        integer, 
    OUT out_o2_id      numeric, 
    OUT out_o2_tr      numeric, 
    OUT out_o2_th      numeric, 
    OUT out_n2_mat_tot numeric,
    OUT out_o2_tm      numeric    -- parallele Personalzeit
) RETURNS record AS $$
  DECLARE 
    op2_rec record;
    uf_r    integer;
    uf_x    integer;
  BEGIN

    SELECT o2_id, o2_zeinh_tr AS zeinhr, o2_zeinh_tx AS zeinhx
    INTO op2_rec
    FROM nk2
      JOIN ab2 ON a2_id = n2_a2_id
      JOIN op2 ON o2_id = a2_o2_id
    WHERE n2_a2_id = in_a2_id;

    out_o2_id   := op2_rec.o2_id;

    out_o2_tr:= (SELECT SUM(n2_ez_stu)
    FROM nk2
    WHERE n2_a2_id = in_a2_id AND n2_ruest);

    out_o2_th = (SELECT SUM(n2_ez_stu)
           FROM nk2
           WHERE n2_a2_id = in_a2_id AND NOT n2_ruest AND NOT n2_pz_para);    -- Hauptzeiten Summe

    out_o2_tm = (SELECT SUM(n2_ez_stu)
           FROM nk2
           WHERE n2_a2_id = in_a2_id AND n2_pz_para);                         -- Parallele Personalzeit Summe    

    out_n2_mat_tot = (
          SELECT SUM(n2_mat)
          FROM nk2
          WHERE 
                n2_a2_id = in_a2_id 
            AND NOT n2_ruest 
            AND NOT n2_pz_para
            AND n2_mat IS NOT NULL
        ); -- gemeldete Menge Total


    IF op2_rec.zeinhr = 1 THEN
      uf_r := 1;
    ELSIF op2_rec.zeinhr = 2 THEN
      uf_r := 60;
    ELSIF op2_rec.zeinhr = 3 THEN
      uf_r := 3600;
    ELSIF op2_rec.zeinhr = 4 THEN
      uf_r := 28800;
    END IF;

    IF op2_rec.zeinhx = 1 THEN
      uf_x := 1;
    ELSIF op2_rec.zeinhx = 2 THEN
      uf_x := 60;
    ELSIF op2_rec.zeinhx = 3 THEN
      uf_x := 3600;
    ELSIF op2_rec.zeinhx = 4 THEN
      uf_x := 28800;
    END IF;

    out_o2_tr := ROUND(out_o2_tr * uf_r, 2);
    out_o2_th := ROUND((out_o2_th / out_n2_mat_tot) * uf_x, 2);       -- Hauptzeit / Totalmenge
    out_o2_tm := round((out_o2_tm / out_n2_mat_tot) * uf_x, 2);       -- parallele Personalzeit

    IF (out_o2_tr IS NOT NULL) AND (out_o2_tr > 0) then
      UPDATE op2 SET o2_tr = out_o2_tr WHERE o2_id = out_o2_id;
    END IF;

    IF (out_o2_th IS NOT NULL) AND (out_o2_th > 0) then
      UPDATE op2 SET o2_th = out_o2_th WHERE o2_id = out_o2_id;
    END IF;

    IF out_o2_tm IS NOT null THEN
      UPDATE op2 SET o2_tm = out_o2_tm WHERE o2_id = out_o2_id;
    END IF;

    RETURN;

 END $$ LANGUAGE plpgsql VOLATILE;


 -- Funktion zur Zuordnung des ersten Arbeitsganges welcher das entsprechende Material mit dem AC-Code benötigt
 -- 14010, 13973, 15135
CREATE OR REPLACE FUNCTION tartikel.stv_op2__upsert_from__op2__by__ksv(
      _stv__st_zn       varchar,
      _akac__stv__st_n  varchar,
      _opl__op_vi       varchar,
      _ksv__ks_abt      varchar
  )  RETURNS void AS $$
  DECLARE
      _rec          record;
      _vorh_sto2_id integer;
  BEGIN

      IF
            _opl__op_vi IS NULL
         OR _stv__st_zn IS NULL
         OR _akac__stv__st_n IS NULL
         OR _ksv__ks_abt IS NULL
      THEN
           -- Zum Aktualisieren werden zwingend die ASK-Variante, der Stücklistenkopfartikel
           -- , der AC-Code sowie die Kostenstelle benötigt.
          RAISE EXCEPTION '%', format( lang_text( 32641 ) );
      END IF;

      FOR _rec IN (
          SELECT st_id, o2_id, op_ix, o2_n
          FROM stv
          -- alle Unterartikel dazuholen
          LEFT JOIN art ON st_n = ak_nr
          -- ASK dazu
          LEFT JOIN opl ON st_zn = op_n
          -- Arbeitsgänge zu ASK und KSV
          LEFT JOIN op2 ON o2_ix = op_ix AND o2_ks = _ksv__ks_abt
          WHERE
                -- Stücklisten-Kopfartikel
                st_zn = _stv__st_zn

                -- Artikelcode der Unteraktikel muss EIngabe entsprechen
            AND ak_ac = _akac__stv__st_n

                -- Arbeitsgangsvariante
            AND op_vi = _opl__op_vi

                -- Ermittlung des ersten Arbeitsganges zur angegebenen Kostenstelle
            AND o2_n = (
                      SELECT min( o2_n )
                        FROM op2
                       WHERE o2_ix = op_ix
                         AND o2_ks = _ksv__ks_abt
            )
      ) LOOP

          -- Ermitteln, eines eventuell schon zugeordneten AG zur STV

          _vorh_sto2_id :=
                sto2_id
                FROM stv_op2
                JOIN op2 ON o2_id = sto2_o2_id
               WHERE sto2_st_id = _rec.st_id
                 AND o2_ix = _rec.op_ix
               LIMIT 1
          ;

          IF _vorh_sto2_id IS NULL THEN

               -- kein vorhanderer Arbeitsgang -> neu einfügen
              INSERT INTO stv_op2 ( sto2_st_id, sto2_o2_id )
              VALUES ( _rec.st_id , _rec.o2_id );

          ELSE

               -- hinterlegen eines anderen Arbeitsganges, wenn Material zu einem anderen Zeitpunkt benötigt wird
              UPDATE stv_op2
                 SET sto2_o2_id = _rec.o2_id
               WHERE sto2_id = _vorh_sto2_id;
          END IF;

      END LOOP;

  END $$ LANGUAGE plpgsql;


-- Upsert für die Zuordnung eines Arbeitsgangs einer ASK-Variante
-- zu einer Stücklistenposition
CREATE OR REPLACE FUNCTION tartikel.stv_op2__upsert_from__op2(
      -- Stücklistenpositions-ID
      _stv__st_id  integer,
      -- ASK Variante
      _opl__op_vi  varchar,
      -- zuzuordnender Arbeitsgang
      _op2__o2_n   integer
  )  RETURNS void AS $$
  DECLARE
      _rec          record;
      _vorh_sto2_id integer;
  BEGIN

      -- Prüfung auf Vollständigkeit der Inputparameter
      IF
            _stv__st_id  IS NULL
         OR _opl__op_vi IS NULL
         OR _op2__o2_n IS NULL
      THEN
           -- Zum Aktualisieren werden zwingend die ASK-Variante, der Stücklistenkopfartikel
           -- , der Stücklistenpositionsartikel sowie die Nummer des Arbeitsgangs benötigt
          RAISE EXCEPTION '%', format( lang_text( 32642 ) );
      END IF;

      -- Ermittlung der o2_id sowie der OP_IX
      FOR _rec in (
          SELECT o2_id, op_ix
          FROM stv
          -- ASK dazu
          JOIN opl ON st_zn = op_n
          -- Arbeitsgänge zu ASK und KSV
          JOIN op2 ON o2_ix = op_ix AND o2_n = _op2__o2_n
          -- Stücklistenpositionsartikel
          WHERE st_id = _stv__st_id
          -- Arbeitsgangsvariante
          AND op_vi = _opl__op_vi
      ) LOOP

          -- Ermitteln, eines eventuell schon zugeordneten AG zur STV
          _vorh_sto2_id :=
                  sto2_id
                  FROM stv_op2
                  JOIN op2 ON o2_id = sto2_o2_id
                 WHERE sto2_st_id = _stv__st_id
                   AND o2_ix = _rec.op_ix
                 LIMIT 1
            ;

          IF _vorh_sto2_id IS NULL THEN

               -- kein vorhanderer Arbeitsgang -> neu einfügen
              INSERT INTO stv_op2 ( sto2_st_id, sto2_o2_id )
              VALUES ( _stv__st_id , _rec.o2_id );

          ELSE

               -- hinterlegen eines anderen Arbeitsganges, wenn Material zu einem anderen Zeitpunkt benötigt wird
              UPDATE stv_op2
                 SET sto2_o2_id = _rec.o2_id
               WHERE sto2_id = _vorh_sto2_id;

          END IF;

      END LOOP;

  END $$ LANGUAGE plpgsql;

-- Upsert für mehrere Positionsartikel einer Sückliste zu einem Arbeitsgang
-- einer ASK-Variante
CREATE OR REPLACE FUNCTION tartikel.stv_op2__upsert_from_list__op2(
      -- kommaseparierte Liste von st_ids
      _stv__st_ids varchar,
      -- ASK Variante
      _opl__op_vi  varchar,
      -- zuzuordnender Arbeitsgang
      _op2__o2_n   integer
  )  RETURNS void AS $$
  DECLARE
      _rec        record;
  BEGIN

      -- Aufruf der Einzelfunktion für alle ST_IDs der Liste
      FOR _rec IN (
        SELECT trim( regexp_split_to_table( _stv__st_ids, ',' ) )::integer AS id
      ) LOOP

          PERFORM tartikel.stv_op2__upsert_from__op2( _rec.id , _opl__op_vi, _op2__o2_n );

      END LOOP;

  END $$ LANGUAGE plpgsql;


-- Löschen der Zuordnung eines Arbeitsgangs einer ASK-Variante
-- zu einem Stücklistenpositionsartikel
CREATE OR REPLACE FUNCTION tartikel.stv_op2__delete(
      -- Stücklistenposition
      _stv__st_id  integer,
      -- ASK Variante
      _opl__op_vi  varchar
  )  RETURNS void AS $$
  DECLARE
      _rec record;
  BEGIN

      -- Prüfung auf Vollständigkeit der Inputparameter
      IF
            _stv__st_id IS NULL
         OR _opl__op_vi IS NULL
      THEN
           -- Zum Aktualisieren werden zwingend die ASK-Variante, der Stücklistenkopfartikel
           -- , der Stücklistenpositionsartikel sowie die Nummer des Arbeitsgangs benötigt
          RAISE EXCEPTION '%', format( lang_text( 32642 ) );
      END IF;

      -- Ermittlung der o2_id
      FOR _rec in (
          SELECT o2_id
          FROM stv
          -- ASK dazu
          JOIN opl ON st_zn = op_n
          -- Arbeitsgänge zu ASK und KSV
          JOIN op2 ON o2_ix = op_ix
          -- Stücklistenpositionsartikel
          WHERE st_id = _stv__st_id
          -- Arbeitsgangsvariante
          AND op_vi = _opl__op_vi
      ) LOOP

          -- Löschen der gefundenen Zuordnung von Arbeitsgang zu Stücklistenpositon
            DELETE FROM stv_op2
                  WHERE sto2_st_id = _stv__st_id
                    AND sto2_o2_id = _rec.o2_id;

      END LOOP;

  END $$ LANGUAGE plpgsql;

-- Löschen der Zuordnung eines Arbeitsgangs einer ASK-Variante
-- zu mehreren Stücklistenpositionsartikeln
CREATE OR REPLACE FUNCTION tartikel.stv_op2__list__delete(
      -- kommaseparierte Liste von st_ids
      _stv__st_ids varchar,
      -- ASK Variante
      _opl__op_vi  varchar
  )  RETURNS void AS $$
  DECLARE
      _rec record;
  BEGIN

      -- Aufruf der Einzelfunktion für alle ST_IDs der Liste
      FOR _rec IN (
        SELECT trim( regexp_split_to_table( _stv__st_ids, ',' ) )::integer AS id
      ) LOOP

          PERFORM tartikel.stv_op2__delete( _rec.id , _opl__op_vi );

      END LOOP;

  END $$ LANGUAGE plpgsql;


-- Funktionen zur Ermittlung der kompletten Kostenstellen-Daten inkl. ggf. abweichender Arbeitspaket-spez. Stundensätze
CREATE OR REPLACE FUNCTION tartikel.ksv__data__by__ks_abt__kssa_ak_nr__get(
      _ks_abt     varchar,        -- Kostenstelle
      _kssa_ak_nr varchar = null  -- Arbeitspaket für spez. Stundensätze
  ) RETURNS ksv AS $$
  DECLARE
      _ks_data    ksv;
      _kssa_data  record;
  BEGIN

      -- Alle Kostenstelle-Daten initial holen.
      _ks_data := ksv FROM ksv WHERE ks_abt = _ks_abt;


      -- Abweichende Arbeitspaket-spezifische Stundensätze ermitteln.
        -- vorherige Prüfung ob vorhanden per index only scan auf ksv_stundensatz_art__kssa_ks_abt__kssa_ak_nr
      IF EXISTS(SELECT true FROM ksv_stundensatz_art WHERE kssa_ks_abt = _ks_abt AND kssa_ak_nr = _kssa_ak_nr) THEN

          SELECT
            kssa_sts, kssa_stsr, kssa_stsm,
            kssa_gss, kssa_gssr, kssa_gssm
          FROM ksv_stundensatz_art
          WHERE kssa_ks_abt = _ks_abt
            AND kssa_ak_nr = _kssa_ak_nr
          INTO _kssa_data
          ;

          -- Abweichende Arbeitspaket-spezifische Stundensätze überschreiben Standard-KS-Stundensätze, wenn nicht NULL.
          _ks_data.ks_sts  := coalesce( _kssa_data.kssa_sts,   _ks_data.ks_sts  );
          _ks_data.ks_stsr := coalesce( _kssa_data.kssa_stsr,  _ks_data.ks_stsr );
          _ks_data.ks_stsm := coalesce( _kssa_data.kssa_stsm,  _ks_data.ks_stsm );
          _ks_data.ks_gss  := coalesce( _kssa_data.kssa_gss,   _ks_data.ks_gss  );
          _ks_data.ks_gssr := coalesce( _kssa_data.kssa_gssr,  _ks_data.ks_gssr );
          _ks_data.ks_gssm := coalesce( _kssa_data.kssa_gssm,  _ks_data.ks_gssm );

      END IF;


      RETURN _ks_data;
  END $$ LANGUAGE plpgsql STABLE;

-- Funktion für op2
CREATE OR REPLACE FUNCTION tartikel.ksv__data__by__table__get(
      _op2 op2
  ) RETURNS ksv AS $$

      SELECT tartikel.ksv__data__by__ks_abt__kssa_ak_nr__get( _op2.o2_ks, _op2.o2_aknr );

  $$ LANGUAGE sql STABLE;

-- Funktion für ab2
CREATE OR REPLACE FUNCTION tartikel.ksv__data__by__table__get(
      _ab2  ab2
  ) RETURNS ksv AS $$

      SELECT tartikel.ksv__data__by__ks_abt__kssa_ak_nr__get( _ab2.a2_ks, _ab2.a2_aknr );

  $$ LANGUAGE sql STABLE;

-- Funktion für bdea
CREATE OR REPLACE FUNCTION tartikel.ksv__data__by__table__get(
      _bdea  bdea
  ) RETURNS ksv AS $$

      SELECT tartikel.ksv__data__by__ks_abt__kssa_ak_nr__get( _bdea.ba_ks, a2_aknr )
      FROM ab2
      WHERE a2_ab_ix = _bdea.ba_ix
      AND a2_n = _bdea.ba_op ;

  $$ LANGUAGE sql STABLE;

-- Funktion für rm
CREATE OR REPLACE FUNCTION tartikel.ksv__data__by__table__get(
      _rm  rm
  ) RETURNS ksv AS $$

      SELECT tartikel.ksv__data__by__ks_abt__kssa_ak_nr__get( _rm.r_ks, a2_aknr )
      FROM ab2
      WHERE a2_id = _rm.r_a2_id;

  $$ LANGUAGE sql STABLE;


-- Funktion für nk2
CREATE OR REPLACE FUNCTION tartikel.ksv__data__by__table__get(
      _nk2  nk2
  ) RETURNS ksv AS $$

      SELECT tartikel.ksv__data__by__ks_abt__kssa_ak_nr__get( _nk2.n2_ks, a2_aknr )
      FROM ab2
      WHERE a2_id = _nk2.n2_a2_id;

  $$ LANGUAGE sql STABLE;


-- Funktion zur Ermittlung des anhand der "ks_kinematic_group" gruppierten eindeutigen Nummernkreises
CREATE OR REPLACE FUNCTION tartikel.ncprogram__new_ncnr__get(
      _o2id      integer, -- ID AG der ASK
      _a2id      integer, -- ID AG der ABK
      _ks        varchar  -- Kostenstelle
  ) RETURNS varchar as $$
  DECLARE
      -- zu ermittelnder maximaler Suffix
      _max_suffix  integer;
      _suffix      integer;
      _vorhanden   boolean;
      -- zu ermittelnde VF-Gruppierung
      _prefix      varchar := coalesce( ks_kinematic_group, ks_abt ) FROM ksv WHERE ks_abt = _ks;
      -- zu ermittelnde NC-Programm-Nummernkreis
      _ncpnr       varchar;
      _ncpid       integer;
  BEGIN

      -- Sicherstellen, dass entweder AG der ASK oder AG der ABK gefüllt
      IF
              _o2id IS NULL
          AND _a2id IS NULL
      THEN
          RAISE EXCEPTION '%', lang_text( 32602 );
      END IF;

      -- ohne gepflegte VF-Gruppe, keine Nummerngenerieriung sinnvoll möglich
      IF
          _prefix IS NULL
      THEN
          RAISE EXCEPTION 'Kostenstelle fehlt xtt17988';
      END IF;


      -- Ermittlung des bisher genutzten Suffix pro VF-Gruppierung, hochzählen um 1
      _max_suffix :=
              coalesce( max( ncp_suffix ) + 1, 1 )
          FROM ncprogram
          WHERE ncp_prefix = _prefix;

      _suffix := _max_suffix;
      LOOP
        -- zusammenfügen der NC-Programm-Bezeichnung aus VF-Gruppierung || Suffix
        -- Suffix soll 5 Stellen umfassen, linksseitiges Auffüllen mit 0
        -- lpad auffüllen mit '0' bis Stellenanzahl 5 erreicht
        _ncpnr := concat( _prefix, lpad( _suffix, 5, '0' ) );
        _suffix := _suffix + 1;
        _vorhanden := EXISTS( SELECT 1 FROM ncprogram WHERE ncp_ncnr = _ncpnr );
        EXIT WHEN NOT _vorhanden OR _suffix >= 100000;
      END LOOP;

      -- eigentlicher INSERT in die ncprogram-Tabelle
      INSERT INTO ncprogram(
          ncp_prefix,
          ncp_suffix,
          ncp_ncnr
      )
      VALUES (
          _prefix,
          _max_suffix,
          _ncpnr
      ) RETURNING ncp_id INTO _ncpid;

      INSERT INTO ncp_o2_a2_ksv (
          ncpoak_ks,
          ncpoak_o2_id,
          ncpoak_a2_id,
          ncpoak_ncp_id
      )
      VALUES (
          _ks,
          _o2id,
          _a2id,
          _ncpid
      );

      --Rückgabe der soeben generierten NC-Nummernkreis an Aufrufer
      RETURN _ncpnr;

 END $$ LANGUAGE plpgsql VOLATILE;
 --


-- Funktion zur Ermittlung des anhand der Kostenstelle eindeutigen Nummernkreises
CREATE OR REPLACE FUNCTION tartikel.messprogram__new_ncnr__get(
      _ks        varchar  -- Kostenstelle
  ) RETURNS varchar as $$
  DECLARE
      -- zu ermittelnder maximaler Suffix
      _max_suffix  integer;
      _suffix      integer;
      _vorhanden   boolean;
      -- zu ermittelnde VF-Gruppierung
      -- #20130 Präfix ist derselbe wie bei den NC-Programmnummern
      _prefix      varchar := coalesce( ks_kinematic_group, ks_abt ) FROM ksv WHERE ks_abt = _ks;
      -- zu ermittelnde NC-Programm-Nummernkreis
      _mspnr       varchar;
  BEGIN

      -- ohne Kostenstelle keine Nummerngenerieriung sinnvoll möglich
      IF
          _prefix IS NULL
      THEN
          RAISE EXCEPTION 'Kostenstelle fehlt xtt17988 %', _ks;
      END IF;


      -- Ermittlung des bisher genutzten Suffix pro VF-Gruppierung, hochzählen um 1
      _max_suffix :=
              coalesce( max( msp_suffix ) + 1, 1 )
          FROM messprogram
          WHERE msp_prefix = _prefix;

      _suffix := _max_suffix;
      LOOP
        -- zusammenfügen der Messprogrammbezeichnung aus Kostenstelle || Suffix
        -- Suffix soll 5 Stellen umfassen, linksseitiges Auffüllen mit 0
        -- lpad auffüllen mit '0' bis Stellenanzahl 5 erreicht
        _mspnr := concat( _prefix, lpad( _suffix, 5, '0' ) );
        _suffix := _suffix + 1;
        _vorhanden := EXISTS( SELECT 1 FROM messprogram WHERE msp_ncnr = _mspnr );
        EXIT WHEN NOT _vorhanden OR _suffix >= 100000;
      END LOOP;

      -- eigentlicher INSERT in die messprogram-Tabelle
      INSERT INTO messprogram(
          msp_prefix,
          msp_suffix,
          msp_ncnr,
          msp_ks
      )
      VALUES (
          _prefix,
          _suffix - 1,
          _mspnr,
          _ks
      );

      --Rückgabe der soeben generierten NC-Nummernkreis an Aufrufer
      RETURN _mspnr;

 END $$ LANGUAGE plpgsql VOLATILE;
 --


-- Funktion zum Korrigieren eines NCProgramms bzw. Neueinfügens
CREATE OR REPLACE FUNCTION tartikel.op2__op2ksa__ncprogram__upsert(
      _o2id   integer,
      _ks     varchar,
      _ncpnr  varchar,
      _ncpid  integer = null
  ) RETURNS void AS $$
  DECLARE
      _ncpid_vorhanden integer;
  BEGIN

      _ncpid_vorhanden := ncp_id
         FROM ncprogram
         JOIN ncp_o2_a2_ksv on ncp_id = ncpoak_ncp_id
         WHERE ncpoak_o2_id = _o2id
           AND ncpoak_ks = _ks
           AND ncp_ncnr = _ncpnr;

      IF  (
             -- ncp_id existiert noch gar nicht
                 SELECT NOT EXISTS( SELECT 1 FROM ncprogram WHERE ncp_id = _ncpid )
             -- ncp_id nicht übergeben
             OR _ncpid is null
          )
          -- es gibt noch keinen Zuordnungseintrag
          AND _ncpid_vorhanden IS NULL
      THEN

          -- einfügen neues NC Program mit Bezeichnung
          INSERT INTO ncprogram ( ncp_ncnr )
          VALUES ( _ncpnr )
          RETURNING ncp_id INTO _ncpid_vorhanden;

          -- Einfügen Zuoprdnung zur o2/ks
          INSERT INTO ncp_o2_a2_ksv (  ncpoak_ncp_id , ncpoak_o2_id,  ncpoak_ks )
          VALUES ( _ncpid_vorhanden, _o2id, _ks );

      ELSE
          -- Nur Update der Bezeichnung
          UPDATE ncprogram
          SET ncp_ncnr = _ncpnr
          WHERE ncp_id = coalesce( _ncpid, _ncpid_vorhanden );

          IF
              EXISTS(
                  SELECT 1
                  FROM ncp_o2_a2_ksv
                  WHERE ncpoak_ks <> _ks
                  AND ncpoak_ncp_id = coalesce( _ncpid, _ncpid_vorhanden )
              )
          THEN

              UPDATE ncp_o2_a2_ksv
              SET ncpoak_ks = _ks
              WHERE ncpoak_ncp_id = coalesce( _ncpid, _ncpid_vorhanden )
              AND ncpoak_o2_id = _o2id;
          END IF;


      END IF;

    RETURN;
END $$ LANGUAGE plpgsql VOLATILE;
--

-- Funktion zum Aktualisieren eines Messprogramms
CREATE OR REPLACE FUNCTION tartikel.op2__ab2__messprogram__upsert(
    _msp_ncnr varchar,        -- Messprogramm, null zeigt an dass Verbindung(en) zum Messprogramm gelöscht werden sollen
    _msp_ks   varchar,        -- Messmaschine
    _o2_id    integer,        -- ASK-AG
    _a2_id    integer         -- ABK-AG
  ) RETURNS void AS $$
  DECLARE
    _msp_id integer;
    _msp_ks_old varchar;
  BEGIN

    -- Keine Arbeitsgänge? Dann ist nichts zu tun.
    IF _o2_id IS null AND _a2_id IS null THEN
      RETURN;
    END IF;

    -- Ist Messprogramm schon bekannt?
    SELECT msp_id, msp_ks INTO _msp_id, _msp_ks_old
    FROM messprogram WHERE msp_ncnr = _msp_ncnr;

    -- Falls ja und Sostenstelle weicht ab, dann diese aktualisieren.
    IF _msp_ks_old <> _msp_ks THEN
      UPDATE messprogram SET msp_ks = _msp_ks WHERE msp_id = _msp_id;
    END IF;

    -- Ggf. Datensatz für Messprogramm neu anlegen
    IF _msp_id IS null AND _msp_ncnr IS NOT null THEN
      INSERT INTO messprogram
        ( msp_ncnr, msp_ks )
      VALUES
        ( _msp_ncnr, _msp_ks )
      RETURNING msp_id INTO _msp_id;
    END IF;

    -- Verbindung ASK-AG->Messprogramm aktualisieren bzw. löschen
    IF _o2_id IS NOT null THEN
      UPDATE op2 SET o2_msp_id = _msp_id WHERE o2_id = _o2_id AND o2_msp_id IS DISTINCT FROM _msp_id; 
    END IF; 

    -- Verbindung ABK-AG->Messprogramm aktualisieren bzw. löschen
    IF _a2_id IS NOT null THEN
      UPDATE ab2 SET a2_msp_id = _msp_id WHERE a2_id = _a2_id AND a2_msp_id IS DISTINCT FROM _msp_id; 
    END IF;     

END $$ LANGUAGE plpgsql VOLATILE;
--

-- Hauptfunktion zum updaten
CREATE OR REPLACE FUNCTION tartikel.op2__ncpoak__ksv__update(
      _o2ks         varchar,
      _o2id         integer,
      _o2ncnr       varchar,
      _o2txt        varchar,
      _o2txtint     varchar,
      _update__op2  boolean,
      _replace__ksv boolean,
      _update__ab2s boolean DEFAULT false,
      _msp_ncnr     varchar DEFAULT null, -- Messprogramm
      _msp_ks       varchar DEFAULT null  -- Messmaschine      
  ) RETURNS void AS $$
  DECLARE
      _o2_ks_alt     varchar(9);
      _o2_ncp_nr_alt varchar(20);
      _o2_ncp_id_alt integer;
      _o2_txt_alt    text;
  BEGIN

      IF
        nullif( _o2ks, '' ) is null
      THEN
        RAISE EXCEPTION 'Kostenstelle fehlt xtt17988 %', _o2ncnr;
      END IF;

      -- bestehende KS/NCNR/TXT-Werte holen von ASK holen
      SELECT o2_ks, o2_nc, o2_txt , ncp_id
      INTO _o2_ks_alt, _o2_ncp_nr_alt, _o2_txt_alt, _o2_ncp_id_alt
      FROM op2
      LEFT JOIN ncprogram ON o2_nc = ncp_ncnr
      WHERE o2_id = _o2id;

      IF
              _o2_ncp_id_alt IS null
          AND _o2_ncp_nr_alt IS NOT null
      THEN

          INSERT INTO ncprogram ( ncp_ncnr )
          VALUES ( _o2_ncp_nr_alt )
          RETURNING ncp_id into _o2_ncp_id_alt;

      END IF;

      -- entsprechende Funktion aufrufen
      IF
          _update__op2
          OR (
                  _o2_ks_alt = _o2ks
              AND _replace__ksv
          )
      THEN
          -- Überarbeiten der o2_nc Nummer und wenn notwendig Ablage und Korrektur einer Alternativ-KS
          PERFORM tartikel.op2__ncpoak__ksv__change( _o2ks, _o2ncnr, _o2txt, _o2txtint, _o2id, _o2_ncp_id_alt, _o2_ncp_nr_alt, _o2_ks_alt, _o2_txt_alt );
      ELSE
          -- sonst: nur alternative KS anlegen und Zuordnung zum ncprogram hinterlegen
          PERFORM tartikel.op2__ncpoak__ksv__alternative( _o2ks, _o2ncnr, _o2txt, _o2txtint, _o2id, _o2_ncp_id_alt, _o2_ks_alt, _o2_txt_alt );
      END IF;

      PERFORM tartikel.op2__ab2__messprogram__upsert( _msp_ncnr, _msp_ks, _o2id, null );

      IF _update__ab2s THEN
          PERFORM tartikel.ab2__a2_ncnr__update( _o2ks, _o2id, _o2ncnr, _o2txt, _o2txtint );

          PERFORM tartikel.op2__ab2__messprogram__upsert( _msp_ncnr, _msp_ks, null, a2_id )
          FROM ab2
          JOIN op2 ON o2_id = _o2id
          JOIN abk ON ab_ix = a2_ab_ix AND ab_askix = o2_ix;
      END IF;
END $$ LANGUAGE plpgsql VOLATILE;


-- Hauptfunktion zum updaten, gibt die Trnsaktion-ID zurück
CREATE OR REPLACE FUNCTION tartikel.op2__ncpoak__ksv__update(
      _o2ks         varchar,
      _o2id         integer,
      _o2ncnr       varchar,
      _o2txt        varchar,
      _o2txtint     varchar,
      _aktion       varchar,    
      _update__ab2s boolean DEFAULT false,
      _msp_ncnr     varchar DEFAULT null, -- Messprogramm
      _msp_ks       varchar DEFAULT null -- Messmaschine        
  ) RETURNS bigint AS $$
  DECLARE
    _update__op2  boolean;
    _replace__ksv boolean;
  BEGIN

     CASE _aktion
         WHEN 'CORRECT' THEN
             _update__op2  := true;
             _replace__ksv := false;
         WHEN 'REPLACE' THEN
             _update__op2  := false;
             _replace__ksv := true;
         WHEN 'ALTERNATIVE' THEN
             _update__op2  := false;
             _replace__ksv := false;
         ELSE
             RAISE EXCEPTION 'Unbekannte auszuführende Aktion % (ncpoak__ksv__update)', upper( _aktion );
     END CASE;

     PERFORM tartikel.op2__ncpoak__ksv__update(
         _o2ks,
         _o2id,
         _o2ncnr,
         _o2txt,
         _o2txtint,
         _update__op2,
         _replace__ksv,
         _update__op2,
         _msp_ncnr, 
         _msp_ks         
     );

     RETURN txid_current();

END $$ LANGUAGE plpgsql VOLATILE;
--

-- Funktion zum updaten der AGs
CREATE OR REPLACE FUNCTION tartikel.ab2__a2_ncnr__update(
      _a2ks         varchar,
      _o2id         integer,
      _a2ncnr       varchar,
      _a2txt        varchar,
      _a2txtint     varchar
  ) RETURNS void as $$

-- Update für NC-Nummer und Texte

-- Das Update erfasst nur Arbeitsgänge, die
--   zur selben ASK gehören wie der übergebenen AG
--   welche dem übergebenen AG zugeordnet sind
--   die Kostenstelle der übergebenen entspricht und
--   weder AG noch ASK beendet wurden.
WITH
  ab2s AS (
    SELECT a2_id FROM ab2
    JOIN abk ON a2_ab_ix = ab_ix
    WHERE a2_ks = _a2ks
      AND a2_o2_id = _o2id
      AND NOT a2_ende
      AND NOT ab_done
  )

  UPDATE ab2 SET
    a2_ncnr = _a2ncnr,
    a2_txt = _a2txt,
    a2_txt_intern = _a2txtint
  WHERE a2_id IN ( SELECT * FROM ab2s )

$$ LANGUAGE sql;

-- Funktion zur Überarbeitung der o2_nc Nummer und eventueller Ablage/ Korrektur einer Alternative
CREATE OR REPLACE FUNCTION tartikel.op2__ncpoak__ksv__change(
      _o2ks          varchar,
      _o2ncnr        varchar,
      _o2txt         text,
      _o2txtint      text,
      _o2id          integer,
      _o2_ncp_id_alt integer,
      _o2_ncp_nr_alt varchar,
      _o2_ks_alt     varchar,
      _o2_txt_alt    text
  ) RETURNS void AS $$
  DECLARE
      _o2_ncid integer;
      _o2_txt_rtf text; --muss zu Übergabeparamter werden
  BEGIN
      _o2_txt_rtf := NULL; --Hotfix: https://redmine.prodat-sql.de/issues/17709#note-19

      -- bei einer eventuell händisch angelegten o2_nc fehlt der Eintrag in die
      -- NCProgrammtabelle -> hier nachholen
      IF _o2ncnr IS NOT null THEN

          INSERT INTO ncprogram ( ncp_prefix, ncp_ncnr )
          SELECT ks_kinematic_group, _o2ncnr
          FROM ksv
          WHERE ks_abt = _o2ks
          ON CONFLICT DO NOTHING
          RETURNING ncp_id INTO _o2_ncid;

      END IF;

      -- wenn kein INSERT möglich - ist das ncprogramm schon vorhanden.
      IF _o2_ncid IS null THEN

          _o2_ncid := ncp_id
              FROM ncprogram
              WHERE ncp_ncnr = _o2ncnr;
      END IF;

      -- wenn in ASK-AG korrigieren oder übernehmen und KS_ALT = KS
      -- Wenn KS gleich
      IF _o2_ks_alt = _o2ks THEN

          -- und ncprogramm Nummer gleich
          IF _o2_ncp_nr_alt = _o2ncnr THEN

              -- Eintrag in op2 bleibt bestehen
              -- nur ergänzen der Texte o2txt und o2txtint
              UPDATE op2
                 SET o2_txt = _o2txt,
                     o2_txt_rtf = _o2_txt_rtf,
                     o2_txt_intern = _o2txtint
               WHERE o2_id = _o2id;


              -- noch aufräumen, falls neue KS/NCNR bereits in KSA vorhanden waren
              DELETE FROM op2ksa
                WHERE o2ks_ks = _o2ks
                  AND o2ks_o2_id = _o2id
                  AND EXISTS(
                        SELECT 1
                        FROM ncp_o2_a2_ksv
                        WHERE ncpoak_o2_id = _o2id
                          AND ncpoak_ks = _o2ks
                          AND ncpoak_ncp_id = _o2_ncp_id_alt
                      )
              ;

          ELSE
              -- wenn NC-Nr. abweichend

              -- Wenn alte NC-Nr. noch nicht in op2ksa existiert -> reinschreiben
              IF _o2_ncp_id_alt IS NOT null  THEN

                  -- Kombination aus o2_id/ks in op2ksa hinzufügen wenn noch nicht vorhanden
                  INSERT INTO op2ksa ( o2ks_ks,  o2ks_o2_id, o2ks_txt )
                  SELECT _o2ks, _o2id, _o2_txt_alt
                  WHERE NOT EXISTS(
                      SELECT 1
                       FROM op2ksa
                      WHERE o2ks_o2_id = _o2id
                        AND o2ks_ks = _o2ks
                  );

                  -- Aktualisieren des Textes einer bereits vorhandenen NCP-Zuordnung zur o2_id/ks-kombination
                  UPDATE ncp_o2_a2_ksv
                    SET ncpoak_txt = _o2_txt_alt
                   WHERE ncpoak_o2_id = _o2id
                     AND ncpoak_ks = _o2ks
                     AND ncpoak_ncp_id = _o2_ncp_id_alt;


                  -- kann passieren wenn o2_nc bisher nur händisch eingetragen
                  INSERT INTO ncp_o2_a2_ksv ( ncpoak_ks, ncpoak_ncp_id, ncpoak_o2_id, ncpoak_txt )
                  SELECT _o2ks, _o2_ncp_id_alt, _o2id, _o2_txt_alt
                  WHERE NOT EXISTS(
                      SELECT 1
                      FROM ncp_o2_a2_ksv
                      WHERE ncpoak_o2_id = _o2id
                      AND ncpoak_ks = _o2ks
                      AND ncpoak_ncp_id = _o2_ncp_id_alt
                  );

              END IF;

              -- Einfügen der Zuordnung des neuen NC Programms
              INSERT INTO ncp_o2_a2_ksv ( ncpoak_ks, ncpoak_ncp_id, ncpoak_o2_id, ncpoak_txt )
              SELECT _o2ks, ncp_id, _o2id, _o2_txt_alt
              FROM ncprogram
              WHERE ncp_ncnr = _o2ncnr;

              -- jetzt die neue NC-Nr. sowie die Texte in op2 schreiben
              UPDATE op2
                 SET o2_nc = _o2ncnr,
                     o2_txt = _o2txt,
                     o2_txt_rtf = _o2_txt_rtf,
                     o2_txt_intern = _o2txtint
               WHERE o2_id = _o2id;

          END IF;

      ELSE
          -- abweichende Kostenstelle

          -- Updaten des Textes einer bestehenden Alternativ-KS
          UPDATE ncp_o2_a2_ksv
          SET ncpoak_txt = _o2_txt_alt
          WHERE ncpoak_o2_id = _o2id
          AND ncpoak_ks = _o2_ks_alt;


          -- Wenn alte NC-Nr. plus KS noch nicht in op2ksa existiert -> reinschreiben
          INSERT INTO op2ksa ( o2ks_ks, o2ks_o2_id, o2ks_txt )
          SELECT _o2_ks_alt, _o2id, _o2_txt_alt
          WHERE NOT EXISTS(
              SELECT 1
              FROM op2ksa
              WHERE o2ks_o2_id = _o2id
                AND o2ks_ks = _o2_ks_alt
          );

          -- im Falle von null (keiner NC nr) kann kein nc-program zuordnung entstehen
          IF _o2_ncp_nr_alt IS NOT null THEN
              INSERT INTO ncp_o2_a2_ksv ( ncpoak_o2_id, ncpoak_ks, ncpoak_ncp_id, ncpoak_txt )
              SELECT _o2id, _o2_ks_alt, _o2_ncp_id_alt, _o2_txt_alt
              WHERE NOT EXISTS(
                  SELECT 1
                  FROM ncp_o2_a2_ksv
                  WHERE ncpoak_o2_id = _o2id
                  AND ncpoak_ks = _o2_ks_alt
                  AND ncpoak_ncp_id = _o2_ncp_id_alt
              );

          END IF;

          --  neue Nc_nummer hinterlegen für ASK
          INSERT INTO ncp_o2_a2_ksv ( ncpoak_ks, ncpoak_ncp_id, ncpoak_o2_id, ncpoak_txt )
          SELECT _o2ks, ncp_id, _o2id, _o2txt
          FROM ncprogram
          WHERE ncp_ncnr = _o2ncnr;

          -- jetzt die neue NC-Nr. sowie die Texte in op2 schreiben
          UPDATE op2
             SET o2_nc = _o2ncnr,
                 o2_ks = _o2ks,
                 o2_txt = _o2txt,
                 o2_txt_rtf = _o2_txt_rtf,
                 o2_txt_intern = _o2txtint
           WHERE o2_id = _o2id;

           -- Bereinigen op2ksa, da übertrag in op2
          DELETE FROM op2ksa
          WHERE o2ks_ks = _o2ks
            AND o2ks_o2_id = _o2id;

      END IF;

END $$ LANGUAGE plpgsql VOLATILE;

-- alternative KS (op2ksa) korrigieren, kein Eingriff in op2
CREATE OR REPLACE FUNCTION tartikel.op2__ncpoak__ksv__alternative(
      _o2ks          varchar,
      _o2ncnr        varchar,
      _o2txt         text,
      _o2txtint      text,
      _o2id          integer,
      _o2_ncp_id_alt integer,
      _o2_ks_alt     varchar,
      _o2_txt_alt    text
  ) RETURNS void AS $$
  DECLARE
      _o2_ncid   integer;
  BEGIN

      INSERT INTO ncprogram ( ncp_prefix, ncp_ncnr )
      SELECT ks_kinematic_group, _o2ncnr
      FROM ksv
      WHERE ks_abt = _o2ks
      ON CONFLICT DO NOTHING
      RETURNING ncp_id INTO _o2_ncid;

      -- wenn kein INSERT möglich - ist das ncprogramm schon vorhanden.
      IF _o2_ncid IS null THEN

          _o2_ncid := ncp_id FROM ncprogram WHERE ncp_ncnr = _o2ncnr;
      END IF;

      -- wenn vorhanden dann update
      UPDATE ncp_o2_a2_ksv
         SET ncpoak_txt = _o2_txt_alt
       WHERE ncpoak_o2_id = _o2id
         AND ncpoak_ks = _o2_ks_alt
         AND ncpoak_ncp_id = _o2_ncp_id_alt;

      -- neue Alternative auch ergänzen
      INSERT INTO op2ksa ( o2ks_ks, o2ks_o2_id, o2ks_txt )
      SELECT _o2ks, _o2id, _o2txt
      WHERE NOT EXISTS(
              SELECT 1
              FROM op2ksa
              WHERE o2ks_o2_id = _o2id
                AND o2ks_ks = _o2ks
            )
        AND NOT EXISTS(
              SELECT 1
              FROM op2
              WHERE o2_ks = _o2ks
                AND o2_nc = _o2ncnr
            )
      ;

      -- Aktualisieren des Textes
      UPDATE ncp_o2_a2_ksv
      SET ncpoak_txt = _o2txt
      WHERE ncpoak_o2_id = _o2id
        AND ncpoak_ks = _o2ks
        AND ncpoak_ncp_id = _o2_ncid;

      -- wenn noch nicht vorhanden Zuordnung noch eintragen
      INSERT INTO ncp_o2_a2_ksv ( ncpoak_ks, ncpoak_ncp_id, ncpoak_o2_id, ncpoak_txt )
      SELECT _o2ks, _o2_ncid, _o2id, _o2txt
      WHERE NOT EXISTS(
          SELECT 1
          FROM ncp_o2_a2_ksv
          WHERE ncpoak_o2_id = _o2id
            AND ncpoak_ks = _o2ks
            AND ncpoak_ncp_id = _o2_ncid
      );

END $$ LANGUAGE plpgsql VOLATILE;
